import { notFound } from 'next/navigation';
import { getProjectBySlug } from '@/lib/storage';
import { templates } from '@/lib/templates';
import { MessageCircle } from 'lucide-react';
import ViewTracker from '@/components/ViewTracker';

// Import Templates
import LandscapingTemplate from '@/components/templates/LandscapingTemplate';
import MovingTemplate from '@/components/templates/MovingTemplate';
import UsedFurnitureTemplate from '@/components/templates/UsedFurnitureTemplate';
import ContractingTemplate from '@/components/templates/ContractingTemplate';
import RecruitmentTemplate from '@/components/templates/RecruitmentTemplate';
import DecorTemplate from '@/components/templates/DecorTemplate';
import PestControlTemplate from '@/components/templates/PestControlTemplate';

export async function generateMetadata({ params }: { params: Promise<{ slug: string }> }) {
    const { slug } = await params;
    const project = await getProjectBySlug(slug);

    if (!project) {
        return {
            title: 'Page Not Found',
        };
    }

    return {
        title: project.data.seoTitle || project.name,
        description: project.data.seoDescription || 'Landing page created with Landing Page Builder',
    };
}

export default async function PublicProjectPage({ params }: { params: Promise<{ slug: string }> }) {
    const { slug } = await params;
    const project = await getProjectBySlug(slug);

    if (!project) {
        notFound();
    }

    const template = templates.find(t => t.id === project.templateId);

    if (!template) {
        return <div>Template not found</div>;
    }

    return (
        <div className="min-h-screen relative">
            {template.id === 'landscaping' && <LandscapingTemplate data={project.data} />}
            {template.id === 'moving' && <MovingTemplate data={project.data} />}
            {template.id === 'used-furniture' && <UsedFurnitureTemplate data={project.data} />}
            {template.id === 'contracting' && <ContractingTemplate data={project.data} />}
            {template.id === 'recruitment' && <RecruitmentTemplate data={project.data} />}
            {template.id === 'decor' && <DecorTemplate data={project.data} />}
            {template.id === 'pest-control' && <PestControlTemplate data={project.data} />}

            <ViewTracker projectId={project.id} />

            {/* Floating WhatsApp Button */}
            {project.data.whatsapp && (
                <a
                    href={`https://wa.me/${project.data.whatsapp}`}
                    target="_blank"
                    className="fixed bottom-6 right-6 bg-green-500 text-white p-4 rounded-full shadow-lg hover:bg-green-600 transition-colors z-50 flex items-center justify-center animate-bounce"
                    title="تواصل معنا عبر واتساب"
                >
                    <MessageCircle className="w-8 h-8" />
                </a>
            )}

            {/* Branding Badge */}
            <div className="fixed bottom-4 left-4 bg-white/90 backdrop-blur shadow-lg rounded-full px-4 py-2 text-xs font-bold text-gray-600 z-40 border border-gray-200">
                تم الإنشاء بواسطة منصة الهبوط
            </div>
        </div>
    );
}
